<?php

namespace App\Http\Controllers;

use App\{
    HoursReport, Company, Equipment, User
};
use Carbon\Carbon;
use Illuminate\Http\Request;
use App\JobRequest;
use Illuminate\Support\Facades\Cache;
use DB;

class RequestController extends Controller
{
    private $jobRequest;

    /**
     * Create a new controller instance.
     *
     * @param JobRequest $jobRequest
     */
    public function __construct(JobRequest $jobRequest)
    {
        $this->jobRequest = $jobRequest;
    }

    /**
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        $req = Cache::remember('job_requests', 23 * 60, function () {
            return $this->jobRequest->latest()->get();
        });

        return response()->json($req, 200);
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function createRequest(Request $request)
    {
        $data = $request->all();
        $lastRequest = $this->jobRequest->latest()
            ->get();

        if ($lastRequest->count() > 0) {
            $lastId = $lastRequest->first()->id;
        } else {
            $lastId = 0;
        }
//
        $engr = collect($data['engineers']);
        $data['engineers'] = json_encode($data['engineers']);
        $data['request_no'] = 'JO-' . Carbon::today()->year . '-' . ($lastId + 1);
        $data['company_id'] = $this->get_company_id($data['customer_name']);
        $data['equipment_id'] = $this->get_equipment_id($data['equipment_sn']);
        $req = JobRequest::create($data);
        $req_id = $req['id'];
        $req->request_desc()->create();
        $engr->map(function ($i) use ($req_id) {
            return HoursReport::create([
                'user_id' => $i['id'],
                'request_id' => $req_id
            ]);
        });
//        if ($req) {
//            return response()->json(['msg' => 'Request Created Successfully'], 201);
//        } else {
//            return response()->json(['msg' => 'Job Request Fail'], 401);
//        }
    }

    /**
     * @param Request $request
     * @param $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateRequest(Request $request, $id)
    {
        $req = JobRequest::findOrFail($id);
        $data = $request->all();
//        $data['engineers'] = json_encode($data['engineers']);

        $status = $req->update($data);
        if ($status) {
            return response()->json(['msg' => 'Request Updated Successfully'], 201);
        } else {
            return response()->json(['msg' => 'Request Update Fail'], 401);
        }

    }

    /**
     * @param $serialNo
     * @return \Illuminate\Http\JsonResponse
     */
    public function latestRequest($serialNo)
    {
        $req = JobRequest::whereEquipment_sn($serialNo)->latest('created_at')->first();
        if (!is_null($req))
            return response()->json($req->created_at, 200);

        return response()->json(['msg' => 'Null'], 404);
    }

    /**
     * @param $serialNo
     * @return \Illuminate\Http\JsonResponse
     */
    public function getJobs($serialNo)
    {
        $req = JobRequest::whereEquipment_sn($serialNo)->latest('created_at')->get();
        try {
            if ($req[0])
                return response()->json($req[0]->created_at, 200);
        } catch (\Exception $exception) {
            return response()->json(['msg' => 'Null'], 404);
        }
    }

    /**
     * @param $serialNo
     * @return \Illuminate\Http\JsonResponse
     */
    public function attendanceFrequency($serialNo)
    {
        $date = Carbon::now()->subMonths(12);
        $total_req = JobRequest::whereEquipment_sn($serialNo)->where(function ($query) use ($date) {
            return $query->where('created_at', '>=', $date)->where('created_at', '<=', Carbon::now());
        })->count('id');
        $res = number_format($total_req / 12, 4);
        return response()->json(['att_frq' => $res], 200);
    }

    /**
     * @return \Illuminate\Http\JsonResponse
     */
    public function getExpenses()
    {
        $jobExpenses = Cache::remember('job_expenses', 23 * 60, function () {
            return $this->jobRequest->jobExpenses()->latest()->get();
        });

        return response()->json($jobExpenses, 200);
    }

    /**
     * @param $id
     * @param $engID
     * @return \Illuminate\Http\JsonResponse
     */
    public function hours_report($id, $engID)
    {
        $report = JobRequest::with([
            'timesheet_items' => function ($query) use ($engID) {
                $query->where('user_id', $engID)->get();
            },
            'job_expenses' => function ($query) use ($engID) {
                $query->where('user_id', $engID)->get();
            }])->where('id', $id)->first();
        $report['company'] = 'LNL';
        $user = User::findOrFail($engID)->profile;
        $company = Company::where('company_name', $report['customer_name'])->first();
        $report['reference'] = $company->reference;
        $report['personal_no'] = $user->staff_id_lnl;
        $report['division'] = $user->division;
        $report['code'] = Equipment::find($report['equipment_id'])->code;
        $report['type'] = Equipment::find($report['equipment_id'])->type;
        $report['series'] = Equipment::find($report['equipment_id'])->series;

        unset($report['engineers']);
        unset($report['created_at']);
        unset($report['updated_at']);
        unset($report['sms_no']);
        unset($report['planing_needed']);
        unset($report['plan_start']);
        unset($report['plan_end']);
        unset($report['urgency']);
        unset($report['comments']);
        unset($report['fleet_number']);
        unset($report['related_quotation']);
        return response()->json($report, 200);
    }

    /**
     * Generate PDF
     * @param $id
     * @param $engID
     * @return \Illuminate\Http\Response
     */
    public function issue_report($id, $engID)
    {
        $report = JobRequest::with([
            'timesheet_items' => function ($query) use ($engID) {
                $query->where('user_id', $engID)->get();
            },
            'job_expenses' => function ($query) use ($engID) {
                $query->where('user_id', $engID)->get();
            }])->where('id', $id)->first();
        $report['company'] = 'LNL';
        $user = User::find($engID)->profile;
        $company = Company::where('company_name', $report['customer_name'])->first();
        $report['reference'] = $company->reference;
        $report['personal_no'] = $user->staff_id_lnl;
        $report['division'] = $user->division;
        $report['code'] = Equipment::find($report['equipment_id'])->code;
        $report['type'] = Equipment::find($report['equipment_id'])->type;
        $report['series'] = Equipment::find($report['equipment_id'])->series;
        unset($report['engineers']);
        unset($report['created_at']);
        unset($report['updated_at']);
        unset($report['sms_no']);
        unset($report['planing_needed']);
        unset($report['plan_start']);
        unset($report['plan_end']);
        unset($report['urgency']);
        unset($report['comments']);
        unset($report['fleet_number']);
        unset($report['related_quotation']);

        $pdf = app()->make('dompdf.wrapper');
        $pdf->loadView('templates.issue_report_pdf', ['report' => $report])->setPaper('a4', 'landscape');
        $pdf->output();
        $dompdf = $pdf->getDomPDF();
        $canvas = $dompdf ->get_canvas();
        $canvas->page_text(700, 25, "Page no. {PAGE_NUM} of {PAGE_COUNT}", null, 10, array(0, 0, 0));
//        return $pdf->stream('issue_report.pdf');
        $pdf->save(base_path('public') . '/issue_report.pdf');

        return response()->json(['file' => $_SERVER['HTTP_HOST'] . '/download-report']);
    }

    public function store_timesheet(Request $request, $req_id)
    {
        $day = $request['day'];
        $month = $request['month']; 
        $year = $request['year'];
        $reportExist = DB::table('timesheet_items')->where([
            ['month', '=', $month],
            ['year', '=', $year],
            ['day', '=', $day]
        ])->count();
        if($reportExist < 1) {
            $req = JobRequest::findOrFail($req_id);
            $req->timesheet_items()->create($request->all());
            return response()->json(['msg' => 'Job Time Created', 'exist' => false], 201);
        } else {
            if($request['type'] == 'travel'){
                $req = JobRequest::findOrFail($req_id);
                $req->timesheet_items()->create($request->all());
                return response()->json(['msg' => 'Job Time Created', 'exist' => false], 201);
            } else {
                return response()->json(['msg' => 'Record for this day already exist, you can update it instead', 'exist' => true], 201);
            }
        }   
    }

    public function update_timesheet(Request $request, $req_id, $timesheet_id)
    {
        $data = $request->all();
        $req = JobRequest::findOrFail($req_id);
        $req->timesheet_items()->findOrFail($timesheet_id)->fill($data)->save();

        return response()->json(['msg' => 'Job Time Updated'], 201);
    }

    public function delete_timesheet($req_id, $timesheet_id)
    {
        // $req = JobRequest::findOrFail($req_id);
        // $req->timesheet_items()->delete($timesheet_id);
        DB::table('timesheet_items')->where('id', $timesheet_id)->delete();
        return response()->json(['msg' => $timesheet_id], 201);
    }

    public function store_expenses(Request $request, $req_id)
    {
        $req = JobRequest::findOrFail($req_id);
        $req->job_expenses()->create($request->all());

        return response()->json(['msg' => 'Job Expenses Created'], 201);
    }

    public function update_expenses(Request $request, $req_id, $exp_id)
    {
        $data = $request->all();
        $req = JobRequest::findOrFail($req_id);
        $req->job_expenses()->findOrFail($exp_id)->fill($data)->save();

        return response()->json(['msg' => 'Job Expenses Updated'], 201);
    }

    public function delete_expenses($req_id, $exp_id)
    {
        $req = JobRequest::findOrFail($req_id);
        $req->job_expenses()->delete($exp_id);

        return response()->json(['msg' => 'Job Expenses Deleted'], 201);
    }

    /**
     * @param $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        JobRequest::destroy($id);
        return response()->json(JobRequest::all());
    }

    private function get_company_id($name)
    {
        $company = Company::where('company_name', $name)->first();
        return $company->id;
    }

    private function get_equipment_id($serial_no)
    {
        $company = Equipment::where('serial_number', $serial_no)->first();
        return $company->id;
    }
}
